// --- CAMBRIDGE ONE CONTENT SCRIPT (v22.11 - Reverted Transcript Logic) ---

// Global State
let fullAutoEnabled = false;
let autoSolveLoopActive = false;

// --- INITIALIZATION & STATE RESTORATION ---
// 1. Check storage on load (This fixes the issue where it stops after page reload)
chrome.storage.sync.get(['fullAutoMode'], (data) => {
    if (data.fullAutoMode) {
        console.log("[Cambridge Solver] 🔄 Auto-mode restored. Resuming...");
        fullAutoEnabled = true;
        // Delay slightly to ensure page stability before restarting loop
        setTimeout(() => {
            if (!autoSolveLoopActive) startAutoSolveEngine();
        }, 1500);
    }
});

// 2. Listen for storage changes (This handles toggling from the popup robustly)
chrome.storage.onChanged.addListener((changes, namespace) => {
    if (namespace === 'sync' && changes.fullAutoMode) {
        const newValue = changes.fullAutoMode.newValue;
        fullAutoEnabled = newValue;
        console.log(`[Cambridge Solver] Auto-mode updated to: ${newValue}`);
        
        if (fullAutoEnabled && !autoSolveLoopActive) {
            startAutoSolveEngine();
        }
    }
});

// --- MESSAGE LISTENER ---
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    // 0. PING (Keep-Alive Check)
    if (request.action === "ping") {
        sendResponse({ success: true, frameId: window.frameElement ? "iframe" : "main" });
        return true;
    }

    // 1. EXTRACT
    if (request.action === "extract_questions") {
        try {
            const data = extractCambridgeContent();
            if (data && data.blanks.length > 0) {
                console.log("[Cambridge Solver] Extraction success:", data.blanks.length, "questions.");
                sendResponse({ success: true, data: data });
            } else {
                sendResponse({ success: false, error: "No questions found in this frame." });
            }
        } catch (e) {
            console.error("[Cambridge Solver] Extraction Error:", e);
            sendResponse({ success: false, error: e.message });
        }
        return true; 
    } 
    
    // 2. APPLY ANSWERS
    else if (request.action === "apply_answers") {
        createFloatingPanel(request.answers);
        
        applyCambridgeAnswersRobust(request.answers).then(count => {
            updateFloatingPanelLog(`Job Complete. Filled ${count} answers.`);
            sendResponse({ success: true, count: count });
        });
        
        return true; 
    }
    
    // 3. AUTO UPDATE
    else if (request.action === "update_auto_solve") {
        fullAutoEnabled = request.enabled;
        if (fullAutoEnabled && !autoSolveLoopActive) startAutoSolveEngine();
        sendResponse({ success: true });
        return true;
    }

    return true; 
});

// --- EXTRACTION ENGINE ---
function extractCambridgeContent() {
    // 1. OPEN TRANSCRIPT (Restored from Alpha Version)
    const transcriptLinks = Array.from(document.querySelectorAll('a, button, span'));
    const showTranscriptBtn = transcriptLinks.find(el => {
        const text = (el.innerText || el.getAttribute('title') || "").toLowerCase();
        return (text.includes('show') || text.includes('transcript')) && el.offsetParent !== null; 
    });
    if (showTranscriptBtn && showTranscriptBtn.innerText.toLowerCase().includes('show')) {
        showTranscriptBtn.click();
    }

    // 2. BUILD CONTEXT
    // We clean the text to avoid excessive whitespace tokens
    let contextText = document.body.innerText.replace(/\s+/g, ' ').trim();
    const blanks = [];
    
    // --- STRATEGY A: Radio Groups ---
    const radioGroups = document.querySelectorAll('.choice_interaction');
    radioGroups.forEach((group, index) => {
        const groupId = group.getAttribute('data-model-id') || `cam_radio_${index}`;
        group.setAttribute('data-solver-uid', groupId); 

        const optionElements = group.querySelectorAll('.is-radiobutton-choice-text, label, .item');
        const options = Array.from(optionElements).map(el => el.innerText.trim()).filter(t => t.length > 0);
        const uniqueOptions = [...new Set(options)];

        if (uniqueOptions.length > 0) {
            blanks.push({ id: groupId, type: 'radio', options: uniqueOptions });
            let questionContext = "Question " + (index + 1);
            const parentP = group.closest('p');
            if (parentP && parentP.previousElementSibling) {
                questionContext = parentP.previousElementSibling.innerText.trim();
            }
            contextText += `\n\n[Question ID: ${groupId}]\nContext: "${questionContext}"\nOptions: [${uniqueOptions.join(', ')}]`;
        }
    });

    // --- STRATEGY F: RICH DROPDOWNS ---
    const richDropdowns = document.querySelectorAll('.wrapper-dropdown');
    if (richDropdowns.length > 0) {
        richDropdowns.forEach((dropdown, index) => {
            const uid = dropdown.getAttribute('data-model-id') || `cam_rich_drop_${index}`;
            dropdown.setAttribute('data-solver-uid', uid);

            // Extract hidden options
            const optionElements = dropdown.querySelectorAll('.popup li');
            const options = Array.from(optionElements).map(el => el.innerText.trim()).filter(t => t.length > 0);

            blanks.push({
                id: uid,
                type: 'rich_select',
                options: options
            });

            // Context Generation using DOM Cloning
            let sentence = "Select option:";
            const parent = dropdown.closest('li') || dropdown.closest('p') || dropdown.closest('.contentblock');
            if (parent) {
                const tempMarker = `__SOLVER_DROP_${index}__`;
                dropdown.setAttribute('data-solver-temp-marker', tempMarker);
                const clone = parent.cloneNode(true);
                const targetInClone = clone.querySelector(`[data-solver-temp-marker="${tempMarker}"]`);
                if (targetInClone) {
                    targetInClone.textContent = " [[TARGET GAP]] ";
                }
                const others = clone.querySelectorAll('.wrapper-dropdown');
                others.forEach(o => { if(o !== targetInClone) o.textContent = " [other gap] "; });

                sentence = clone.innerText.replace(/\s+/g, ' ').trim();
                dropdown.removeAttribute('data-solver-temp-marker');
            }
            contextText += `\n[Dropdown ID: ${uid}]\nContext: "${sentence}"\nOptions: [${options.join(', ')}]`;
        });
    }

    // --- STRATEGY E: DRAG & DROP ---
    const draggables = document.querySelectorAll('.gap_match_drag, .drag_element, .draggable, .source-item');
    const dropZones = document.querySelectorAll('.gap_match_gap_view, .dropzone, .target');

    // Filter valid zones
    const validDropZones = Array.from(dropZones).filter(el => {
        const isInSourcePool = el.closest('.pool') || el.closest('.source-container');
        const isSourceSlot = el.classList.contains('has_drag');
        const isMeasure = el.closest('.width-measure'); 
        return !isInSourcePool && !isSourceSlot && !isMeasure;
    });

    if (draggables.length > 0 && validDropZones.length > 0) {
        const draggableWords = Array.from(draggables)
                                    .filter(d => !d.closest('.width-measure')) 
                                    .map(d => d.innerText.replace(/\s+/g, '').trim())
                                    .filter(t => t.length > 0);
        const uniqueWords = [...new Set(draggableWords)];
        
        contextText += `\n\n[WORD BANK]: ${uniqueWords.join(', ')}\n`;
        contextText += `INSTRUCTION: Match the words from the [WORD BANK] to the [TARGET GAP] markers below.\n`;

        validDropZones.forEach((zone, index) => {
            const uid = `cam_dnd_${index}`;
            const modelId = zone.getAttribute('data-model-id');
            const finalId = modelId || uid;
            
            zone.setAttribute('data-solver-uid', finalId);
            
            blanks.push({
                id: finalId,
                type: 'dnd',
                options: uniqueWords
            });

            let sentence = "Fill in the blank:";
            const parent = zone.closest('li') || zone.closest('p') || zone.closest('.contentblock');
            
            if (parent) {
                const tempMarker = `__SOLVER_TARGET_${index}__`;
                zone.setAttribute('data-solver-temp-marker', tempMarker);
                const clone = parent.cloneNode(true);
                const targetInClone = clone.querySelector(`[data-solver-temp-marker="${tempMarker}"]`);
                if (targetInClone) targetInClone.textContent = " [[TARGET GAP]] ";
                const otherZones = clone.querySelectorAll('.gap_match_gap_view, .dropzone, .target');
                otherZones.forEach(other => { if (other !== targetInClone) other.textContent = " [other gap] "; });
                sentence = clone.innerText.replace(/\s+/g, ' ').trim();
                zone.removeAttribute('data-solver-temp-marker');
            }
            contextText += `\n[Gap ID: ${finalId}]\nContext: "${sentence}" (Select from Word Bank)`;
        });
    }

    // --- STRATEGY B: Text Gaps / Number Inputs ---
    const gapInputs = document.querySelectorAll('.input-text input, .ic-text-gap input, input[type="number"]');
    let numberInputFound = false;
    
    gapInputs.forEach((input, index) => {
        if (input.type === 'hidden') return;
        const wrapper = input.closest('.input-text') || input;
        const modelId = wrapper.getAttribute('data-model-id');
        const uid = modelId || `cam_gap_${index}`;
        
        input.setAttribute('data-solver-uid', uid);
        if (wrapper !== input) wrapper.setAttribute('data-solver-uid-backup', uid);

        if (blanks.some(b => b.id === uid)) return;
        if (input.type === 'number') numberInputFound = true;

        blanks.push({
            id: uid,
            type: input.type === 'number' ? 'number' : 'text',
            options: [] 
        });

        // Context & Placeholder Hint
        let sentence = "Fill in the blank:";
        const parentContainer = input.closest('li') || input.closest('p') || input.closest('.contentblock');
        const placeholder = input.getAttribute('placeholder');
        let hintText = "";
        if (placeholder && placeholder.trim().length > 0) {
            hintText = ` (Constraint: Starts with/Format "${placeholder}")`;
        }

        if (parentContainer) {
            const tempMarker = `__SOLVER_INPUT_${index}__`;
            input.setAttribute('data-solver-temp-marker', tempMarker);
            const clone = parentContainer.cloneNode(true);
            const targetInClone = clone.querySelector(`[data-solver-temp-marker="${tempMarker}"]`);
            if (targetInClone) {
                const wrapperInClone = targetInClone.closest('.input-text') || targetInClone;
                const marker = document.createTextNode(` [[TARGET GAP]] `);
                wrapperInClone.parentNode.replaceChild(marker, wrapperInClone);
            }
            // Mask others
            const otherInputs = clone.querySelectorAll('input');
            otherInputs.forEach(other => {
                const wrapperOther = other.closest('.input-text') || other;
                if (wrapperOther.parentNode) wrapperOther.parentNode.replaceChild(document.createTextNode(" [other gap] "), wrapperOther);
            });

            sentence = clone.innerText.replace(/\s+/g, ' ').trim() + hintText;
            input.removeAttribute('data-solver-temp-marker');
        }
        contextText += `\n[Gap ID: ${uid}]\nContext: "${sentence}"`;
    });

    if (numberInputFound) {
        contextText = "IMPORTANT: For the numbered gaps, provide the sequence order (1, 2, 3...) based on the text.\n\n" + contextText;
    }

    // --- STRATEGY C: Standard Inputs ---
    const standardInputs = document.querySelectorAll('input[type="text"]:not([data-solver-uid]), textarea:not([data-solver-uid])');
    standardInputs.forEach((input, index) => {
        const uid = `cam_input_${index}`;
        input.setAttribute('data-solver-uid', uid);
        blanks.push({ id: uid, type: 'text', options: [] });
        contextText += `\n[Gap ${uid}]`;
    });

    // --- STRATEGY D: Dropdowns ---
    const selects = document.querySelectorAll('select');
    selects.forEach((select, index) => {
        const uid = `cam_select_${index}`;
        select.setAttribute('data-solver-uid', uid);
        const opts = Array.from(select.options).filter(o => o.value).map(o => o.text);
        blanks.push({ id: uid, type: 'select', options: opts });
        contextText += `\n[Dropdown ${uid}: ${opts.join('/')}]`;
    });

    return { text: contextText, blanks: blanks };
}

// --- ROBUST ASYNC APPLIER ---
async function applyCambridgeAnswersRobust(answers) {
    let count = 0;
    const entries = Object.entries(answers);
    const textQueue = [];
    const dndQueue = [];
    const dropdownQueue = [];

    // 1. CLASSIFICATION
    for (const [uid, ans] of entries) {
        if (!ans) continue;

        const el = document.querySelector(`[data-solver-uid="${uid}"]`) || 
                   document.querySelector(`[data-model-id="${uid}"]`);

        // Check for Rich Dropdown
        if (el && el.classList.contains('wrapper-dropdown')) {
            dropdownQueue.push({ uid, ans });
            continue;
        }

        if (el && (el.classList.contains('gap_match_gap_view') || el.classList.contains('dropzone') || el.classList.contains('target'))) {
            dndQueue.push({ uid, ans });
            continue;
        }
        if (uid.includes('cam_dnd_')) {
            dndQueue.push({ uid, ans });
            continue;
        }

        if (el) {
            if (el.classList.contains('choice_interaction') || el.getAttribute('role') === 'radiogroup') {
                applyRadio(el, ans);
                count++;
                continue; 
            }
            if (el.tagName === 'SELECT') {
                applySelect(el, ans);
                count++;
                continue;
            }
        }
        textQueue.push({ uid, ans });
    }

    // 2. PROCESS RICH DROPDOWNS (Click -> Select)
    for (let i = 0; i < dropdownQueue.length; i++) {
        const item = dropdownQueue[i];
        try {
            const dropdown = document.querySelector(`[data-solver-uid="${item.uid}"]`) || 
                             document.querySelector(`[data-model-id="${item.uid}"]`);
            
            if (dropdown) {
                updateFloatingPanelLog(`Selecting "${item.ans}"...`);
                dropdown.scrollIntoView({ behavior: "smooth", block: "center" });
                
                // 1. Click Trigger (usually .rich-dropdown or .drop-label)
                const trigger = dropdown.querySelector('.rich-dropdown, .drop-label');
                if (trigger) {
                    trigger.click();
                    await new Promise(r => setTimeout(r, 300)); // Wait for animation
                    
                    // 2. Find Option in Popup
                    const options = Array.from(dropdown.querySelectorAll('.popup li'));
                    const cleanAnswer = item.ans.toString().trim().toLowerCase();
                    const target = options.find(o => o.innerText.trim().toLowerCase() === cleanAnswer);
                    
                    if (target) {
                        target.click();
                        dropdown.style.border = "2px solid #00e676"; // Success Green
                        updateFloatingPanelLog(`Selected: ${item.ans}`);
                        count++;
                    } else {
                        updateFloatingPanelLog(`Err: Option not found "${item.ans}"`);
                    }
                    // Close popup safety
                    await new Promise(r => setTimeout(r, 200));
                }
            }
        } catch(e) { console.error(e); }
    }

    // 3. PROCESS DRAG & DROP (MULTI-POOL)
    for (let i = 0; i < dndQueue.length; i++) {
        const item = dndQueue[i];
        try {
            const dropZone = document.querySelector(`[data-solver-uid="${item.uid}"]`);
            
            // Re-scan pools every iteration
            const pools = document.querySelectorAll('.pool, .wordpool');
            let draggables = [];
            const seenElements = new Set();

            const addDraggables = (nodeList) => {
                nodeList.forEach(d => {
                    if (d.closest('.width-measure')) return;
                    if (seenElements.has(d)) return;
                    if (isElementInteractable(d)) {
                        draggables.push(d);
                        seenElements.add(d);
                    }
                });
            };

            pools.forEach(pool => addDraggables(pool.querySelectorAll('.gap_match_drag, .drag_element, .draggable')));
            // Global fallback
            addDraggables(document.querySelectorAll('.gap_match_drag, .drag_element, .draggable'));

            const cleanTarget = item.ans.toString().toLowerCase().replace(/\s+/g, '');
            
            // Find Source
            const source = draggables.find(d => {
                const innerT = (d.innerText || "").toLowerCase().replace(/\s+/g, '');
                const textC = (d.textContent || "").toLowerCase().replace(/\s+/g, '');
                return innerT === cleanTarget || innerT.includes(cleanTarget) || 
                       textC === cleanTarget || textC.includes(cleanTarget);
            });

            if (dropZone && source) {
                // Focus Accessibility Hack
                const sourceFocusable = source.matches('[tabindex="0"]') ? source : source.querySelector('[tabindex="0"]');
                const zoneFocusable = dropZone.matches('[tabindex="0"]') ? dropZone : dropZone.querySelector('[tabindex="0"]');
                
                const targetSource = sourceFocusable || source;
                const targetZone = zoneFocusable || dropZone;

                if (targetSource && targetZone) {
                    updateFloatingPanelLog(`Dragging "${item.ans}"...`);
                    targetSource.scrollIntoView({ behavior: "auto", block: "nearest" });
                    
                    // Activate Source
                    targetSource.focus();
                    await new Promise(r => setTimeout(r, 50));
                    if (document.activeElement !== targetSource) { targetSource.click(); targetSource.focus(); }
                    await new Promise(r => setTimeout(r, 50));

                    if (document.activeElement === targetSource || targetSource.classList.contains('selected') || targetSource.closest('.selected')) {
                         simulateEnterKey(targetSource);
                         source.style.outline = "4px solid #ff9100";
                         await new Promise(r => setTimeout(r, 200));

                         // Activate Zone
                         targetZone.scrollIntoView({ behavior: "auto", block: "nearest" });
                         targetZone.focus();
                         await new Promise(r => setTimeout(r, 50));

                         if (document.activeElement === targetZone) {
                             simulateEnterKey(targetZone);
                             dropZone.style.outline = "4px solid #ff9100";
                             updateFloatingPanelLog(`Dropped: ${item.ans}`);
                             count++;
                         } else {
                             targetZone.click(); // Force click
                             count++;
                         }
                    }
                    await new Promise(r => setTimeout(r, 300));
                    source.style.outline = "";
                    dropZone.style.outline = "";
                }
            }
        } catch (e) { console.error(e); }
    }

    // 4. PROCESS TEXT INPUTS
    for (let i = 0; i < textQueue.length; i++) {
        const item = textQueue[i];
        const answerVal = item.ans.toString();
        let targetEl = null;

        for (let attempt = 0; attempt < 3; attempt++) {
            targetEl = document.querySelector(`input[data-solver-uid="${item.uid}"]`);
            if (targetEl) break;
            
            const wrapper = document.querySelector(`[data-model-id="${item.uid}"]`);
            if (wrapper) { targetEl = wrapper.querySelector('input'); if(targetEl) break; }

            await new Promise(r => setTimeout(r, 100));
        }

        if (targetEl) {
            try {
                targetEl.scrollIntoView({ behavior: "smooth", block: "center" });
                targetEl.style.border = "3px solid #00e676"; 
                const wrapper = targetEl.closest('.input-text');
                if (wrapper) wrapper.classList.add('filled');

                if (wrapper && wrapper.classList.contains('characters-in-gap')) {
                    await fillInputCharByChar(targetEl, answerVal);
                } else {
                    await fillInputSequence(targetEl, answerVal);
                }
                count++;
                await new Promise(r => setTimeout(r, 200));
            } catch (err) { console.error(err); }
        }
    }

    return count;
}

// --- FILLER HELPERS ---

function simulateEnterKey(element) {
    const keyEvents = ['keydown', 'keypress', 'keyup'];
    keyEvents.forEach(type => {
        element.dispatchEvent(new KeyboardEvent(type, {
            key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true
        }));
    });
}

function applyRadio(element, answer) {
    const cleanAnswer = answer.toString().trim().toLowerCase();
    const candidates = Array.from(element.querySelectorAll('.is-radiobutton-choice-text, label, .item'));
    let target = candidates.find(el => el.innerText.trim().toLowerCase() === cleanAnswer) || 
                 candidates.find(el => el.innerText.trim().toLowerCase().includes(cleanAnswer));
    
    if (target) {
        const radioInput = target.closest('.input-radio') || target;
        radioInput.click();
        const indicator = radioInput.querySelector('input');
        if (indicator) indicator.style.border = "3px solid #00e676";
        else radioInput.style.border = "2px solid #00e676";
    }
}

function applySelect(element, answer) {
    const opts = Array.from(element.options);
    const match = opts.find(o => o.text.trim() === answer.toString().trim());
    if (match) {
        element.value = match.value;
        element.dispatchEvent(new Event('change', { bubbles: true }));
    }
}

async function fillInputSequence(element, value) {
    element.focus();
    element.click();
    await new Promise(r => setTimeout(r, 50));
    
    const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, "value").set;
    nativeSetter.call(element, '');
    element.dispatchEvent(new Event('input', { bubbles: true }));
    
    const success = document.execCommand('insertText', false, value);

    if (!success || element.value !== value) {
        nativeSetter.call(element, value);
    }

    element.dispatchEvent(new Event('input', { bubbles: true }));
    element.dispatchEvent(new Event('change', { bubbles: true }));
    element.dispatchEvent(new KeyboardEvent('keyup', { bubbles: true, key: value.toString().slice(-1) }));
    element.dispatchEvent(new Event('blur', { bubbles: true }));
}

async function fillInputCharByChar(element, value) {
    element.focus();
    element.click();
    await new Promise(r => setTimeout(r, 50));
    
    const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, "value").set;
    nativeSetter.call(element, '');
    element.dispatchEvent(new Event('input', { bubbles: true }));

    for (let i = 0; i < value.length; i++) {
        const char = value[i];
        // Simulate real key events
        element.dispatchEvent(new KeyboardEvent('keydown', { key: char, code: `Key${char.toUpperCase()}`, bubbles: true }));
        element.dispatchEvent(new KeyboardEvent('keypress', { key: char, charCode: char.charCodeAt(0), bubbles: true }));
        
        const prevVal = element.value;
        const nextVal = prevVal + char;
        nativeSetter.call(element, nextVal);

        element.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'insertText', data: char }));
        element.dispatchEvent(new KeyboardEvent('keyup', { key: char, code: `Key${char.toUpperCase()}`, bubbles: true }));

        await new Promise(r => setTimeout(r, 40));
    }
    element.dispatchEvent(new Event('change', { bubbles: true }));
    element.dispatchEvent(new Event('blur', { bubbles: true }));
}

function isElementInteractable(el) {
    if (!el) return false;
    if (el.closest('.pool, .wordpool')) return !el.closest('.width-measure');
    return !!(el.offsetWidth || el.offsetHeight || el.getClientRects().length);
}

// --- UI HELPERS ---
function createFloatingPanel(answers) {
    const oldPanel = document.getElementById('cam-solver-panel');
    if (oldPanel) oldPanel.remove();

    const panel = document.createElement('div');
    panel.id = 'cam-solver-panel';
    panel.style.cssText = `
        position: fixed; top: 60px; right: 20px; width: 300px;
        background: #1e1e1e; border: 1px solid #333; color: #fff;
        z-index: 2147483647; border-radius: 8px; font-family: sans-serif;
        box-shadow: 0 4px 15px rgba(0,0,0,0.5); font-size: 13px;
        display: flex; flex-direction: column;
    `;

    const header = document.createElement('div');
    header.innerHTML = '<strong>🤖 Solver Progress</strong> <span style="float:right;cursor:pointer" onclick="this.parentElement.parentElement.remove()">✕</span>';
    header.style.cssText = 'padding: 10px; border-bottom: 1px solid #333; background: #252525;';
    panel.appendChild(header);

    const logContainer = document.createElement('div');
    logContainer.id = 'cam-solver-logs';
    logContainer.style.cssText = 'padding: 5px; background: #000; color: #00e676; font-family: monospace; font-size: 10px; height: 100px; overflow-y: auto; border-bottom: 1px solid #333;';
    panel.appendChild(logContainer);

    const list = document.createElement('div');
    list.style.padding = '10px';
    list.style.maxHeight = '300px';
    list.style.overflowY = 'auto';

    let i = 1;
    for (const [id, ans] of Object.entries(answers)) {
        const item = document.createElement('div');
        item.innerHTML = `<span style="color:#1793d1; font-weight:bold;">${i}.</span> <span style="user-select:all">${ans}</span>`;
        item.style.marginBottom = '4px';
        item.style.borderBottom = '1px solid #333';
        list.appendChild(item);
        i++;
    }
    panel.appendChild(list);
    document.body.appendChild(panel);
}

function updateFloatingPanelLog(msg) {
    const logs = document.getElementById('cam-solver-logs');
    if (logs) {
        const line = document.createElement('div');
        line.textContent = `> ${msg}`;
        logs.appendChild(line);
        logs.scrollTop = logs.scrollHeight;
    }
}

// --- AUTO SOLVE LOOP ---
async function startAutoSolveEngine() {
    autoSolveLoopActive = true;
    while (fullAutoEnabled && autoSolveLoopActive) {
        try {
            if (document.readyState !== 'complete') { await new Promise(r => setTimeout(r, 1000)); continue; }
            
            const extraction = extractCambridgeContent(); 
            if (extraction && extraction.blanks.length > 0) {
                const response = await chrome.runtime.sendMessage({
                    action: "solve_securely",
                    payload: extraction,
                    autoMode: true
                });
                if (response && response.success) {
                    createFloatingPanel(response.answers);
                    await applyCambridgeAnswersRobust(response.answers);
                    
                    await new Promise(r => setTimeout(r, 2000));
                    const checkBtn = Array.from(document.querySelectorAll('button')).find(b => b.innerText.match(/Check|Submit/i));
                    if(checkBtn) checkBtn.click();
                    await new Promise(r => setTimeout(r, 3000));
                }
            }
            const nextBtn = document.querySelector('a.next, button[title="Next"]');
            if (nextBtn) nextBtn.click();
            else break;
            
            await new Promise(r => setTimeout(r, 4000));
        } catch (e) {
            console.error(e);
            break;
        }
    }
    autoSolveLoopActive = false;
}